import React, { createContext, useContext, useState, useEffect, ReactNode } from 'react';
import { auth } from '../firebase'; // Adjust path if your firebase.ts is elsewhere
import { 
  User, 
  onAuthStateChanged,
  createUserWithEmailAndPassword, // We'll use this for signup
  signInWithEmailAndPassword,    // We'll use this for login
  signOut,                       // We'll use this for logout
  // Other Firebase auth functions can be imported as needed
} from 'firebase/auth';

// Define the shape of the context data
interface AuthContextType {
  currentUser: User | null;
  loading: boolean;
  signup: (email: string, password: string) => Promise<any>;
  login: (email: string, password: string) => Promise<any>;  // Updated signature
  logout: () => Promise<void>;
}

// Create the context with a default undefined value (or a default object)
// Using 'undefined' initially and checking in useAuth is a common pattern
const AuthContext = createContext<AuthContextType | undefined>(undefined);

// Custom hook to use the auth context
export function useAuth() {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
}

// Define props for the AuthProvider
interface AuthProviderProps {
  children: ReactNode;
}

// Create the AuthProvider component
export const AuthProvider: React.FC<AuthProviderProps> = ({ children }) => {
  const [currentUser, setCurrentUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true); // Start with loading true

  useEffect(() => {
    // Listen for authentication state changes
    const unsubscribe = onAuthStateChanged(auth, (user) => {
      setCurrentUser(user);
      setLoading(false); // Set loading to false once we have a user or know there isn't one
    });

    // Cleanup subscription on unmount
    return unsubscribe;
  }, []);

  // Implement the signup function
  const signup = async (email: string, password: string) => {
    setLoading(true); // Optionally set loading true during signup
    try {
      const userCredential = await createUserWithEmailAndPassword(auth, email, password);
      // You can do something with userCredential.user here if needed
      // e.g., set up a user profile in Firestore/Realtime Database
      setLoading(false);
      return userCredential;
    } catch (error) {
      setLoading(false);
      // Handle errors here (e.g., display to user, log to console)
      console.error("Error signing up:", error);
      throw error; // Re-throw the error so the calling component can handle it
    }
  };

  // Implement the login function
  const login = async (email: string, password: string) => {
    setLoading(true); // Optionally set loading true during login
    try {
      const userCredential = await signInWithEmailAndPassword(auth, email, password);
      // You can do something with userCredential.user here if needed
      setLoading(false);
      return userCredential;
    } catch (error) {
      setLoading(false);
      // Handle errors here (e.g., display to user, log to console)
      console.error("Error logging in:", error);
      throw error; // Re-throw the error so the calling component can handle it
    }
  };

  const logout = async () => {
    console.log('Logout function called');
    return signOut(auth); // Firebase signOut is straightforward
  };

  const value = {
    currentUser,
    loading,
    signup,
    login,
    logout,
  };

  // Provide the context value to children components
  // Don't render children until loading is false to prevent rendering protected routes prematurely
  return (
    <AuthContext.Provider value={value}>
      {!loading && children}
    </AuthContext.Provider>
  );
}; 